---
title: Work with JavaScript objects, methods, functions and globals from Python
  Workers · Cloudflare Workers docs
description: "Via Pyodide, Python Workers provide a Foreign Function Interface
  (FFI) to JavaScript. This allows you to:"
lastUpdated: 2025-09-01T10:19:51.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/workers/languages/python/ffi/
  md: https://developers.cloudflare.com/workers/languages/python/ffi/index.md
---

Via [Pyodide](https://pyodide.org/en/stable/), Python Workers provide a [Foreign Function Interface (FFI)](https://en.wikipedia.org/wiki/Foreign_function_interface) to JavaScript. This allows you to:

* Use [bindings](https://developers.cloudflare.com/workers/runtime-apis/bindings/) to resources on Cloudflare, including [Workers AI](https://developers.cloudflare.com/workers-ai/), [Vectorize](https://developers.cloudflare.com/vectorize/), [R2](https://developers.cloudflare.com/r2/), [KV](https://developers.cloudflare.com/kv/), [D1](https://developers.cloudflare.com/d1/), [Queues](https://developers.cloudflare.com/queues/), [Durable Objects](https://developers.cloudflare.com/durable-objects/), [Service Bindings](https://developers.cloudflare.com/workers/runtime-apis/bindings/service-bindings/) and more.
* Use JavaScript globals, like [`Request`](https://developers.cloudflare.com/workers/runtime-apis/request/), [`Response`](https://developers.cloudflare.com/workers/runtime-apis/response/), and [`fetch()`](https://developers.cloudflare.com/workers/runtime-apis/fetch/).
* Use the full feature set of Cloudflare Workers — if an API is accessible in JavaScript, you can also access it in a Python Worker, writing exclusively Python code.

The details of Pyodide's Foreign Function Interface are documented [here](https://pyodide.org/en/stable/usage/type-conversions.html), and Workers written in Python are able to take full advantage of this.

## Using Bindings from Python Workers

Bindings allow your Worker to interact with resources on the Cloudflare Developer Platform. When you declare a binding on your Worker, you grant it a specific capability, such as being able to read and write files to an [R2](https://developers.cloudflare.com/r2/) bucket.

For example, to access a [KV](https://developers.cloudflare.com/kv) namespace from a Python Worker, you would declare the following in your Worker's [Wrangler configuration file](https://developers.cloudflare.com/workers/wrangler/configuration/):

* wrangler.jsonc

  ```jsonc
  {
    "main": "./src/index.py",
    "kv_namespaces": [
      {
        "binding": "FOO",
        "id": "<YOUR_KV_NAMESPACE_ID>"
      }
    ]
  }
  ```

* wrangler.toml

  ```toml
  main = "./src/index.py"
  kv_namespaces = [
    { binding = "FOO", id = "<YOUR_KV_NAMESPACE_ID>" }
  ]
  ```

...and then call `.get()` on the binding object that is exposed on `env`:

```python
from workers import WorkerEntrypoint, Response


class Default(WorkerEntrypoint):
    async def fetch(self, request):
        await self.env.FOO.put("bar", "baz")
        bar = await self.env.FOO.get("bar")
        return Response(bar) # returns "baz"
```

Under the hood, `env` is actually a JavaScript object. When you call `.FOO`, you are accessing this property via a [`JsProxy`](https://pyodide.org/en/stable/usage/api/python-api/ffi.html#pyodide.ffi.JsProxy) — special proxy object that makes a JavaScript object behave like a Python object.

## Using JavaScript globals from Python Workers

When writing Workers in Python, you can access JavaScript globals by importing them from the `js` module. For example, note how `Response` is imported from `js` in the example below:

```python
from workers import WorkerEntrypoint
from js import Response


class Default(WorkerEntrypoint):
    async def fetch(self, request):
        return Response.new("Hello World!")
```

Refer to the [Python examples](https://developers.cloudflare.com/workers/languages/python/examples/) to learn how to call into JavaScript functions from Python, including `console.log` and logging, providing options to `Response`, and parsing JSON.
