---
title: Common HTTP policies · Cloudflare Zero Trust docs
description: The following policies are commonly used to secure HTTP traffic.
lastUpdated: 2025-10-22T21:11:06.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/cloudflare-one/traffic-policies/http-policies/common-policies/
  md: https://developers.cloudflare.com/cloudflare-one/traffic-policies/http-policies/common-policies/index.md
---

The following policies are commonly used to secure HTTP traffic.

Refer to the [HTTP policies page](https://developers.cloudflare.com/cloudflare-one/traffic-policies/http-policies/) for a comprehensive list of other selectors, operators, and actions.

## Block sites

Block attempts to reach sites by hostname or URL paths. Different approaches may be required based on how a site is organized.

### Block sites by hostname

Block all subdomains that use a host.

* Dashboard

  | Selector | Operator | Value | Action |
  | - | - | - | - |
  | Host | matches regex | `.*example\.com` | Block |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Block sites by hostname",
      "description": "Block all subdomains that use a specific hostname",
      "enabled": true,
      "action": "block",
      "filters": [
          "http"
      ],
      "traffic": "http.request.host matches \".*example.com\"",
      "identity": "",
      "device_posture": ""
    }'
  ```

### Block sites by URL

Block a section of a site without blocking the entire site. For example, you can block a specific subreddit, such as `reddit.com/r/gaming`, without blocking `reddit.com`.

* Dashboard

  | Selector | Operator | Value | Action |
  | - | - | - | - |
  | URL | matches regex | `/r/gaming` | Block |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Block sites by URL",
      "description": "Block specific parts of a site without blocking the hostname",
      "enabled": true,
      "action": "block",
      "filters": [
          "http"
      ],
      "traffic": "http.request.uri matches \"/r/gaming\"",
      "identity": "",
      "device_posture": ""
    }'
  ```

## Block content categories

Block content categories which go against your organization's acceptable use policy.

* Dashboard

  | Selector | Operator | Value | Action |
  | - | - | - | - |
  | Content Categories | in | *Questionable Content*, *Security Risks*, *Miscellaneous*, *Adult Themes*, *Gambling* | Block |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "All-HTTP-ContentCategories-Blocklist",
      "description": "Block access to questionable content and potential security risks",
      "precedence": 40,
      "enabled": true,
      "action": "block",
      "filters": [
          "http"
      ],
      "traffic": "any(http.request.uri.content_category[*] in {17 85 87 102 157 135 138 180 162 32 169 177 128 15 115 119 124 141 161 2 67 125 133 99})",
      "identity": "",
      "device_posture": ""
    }'
  ```

* Terraform

  ```tf
  resource "cloudflare_zero_trust_gateway_policy" "block_unauthorized_apps" {
    account_id     = var.cloudflare_account_id
    name           = "All-HTTP-ContentCategories-Blocklist"
    description    = "Block access to questionable content and potential security risks"
    precedence     = 40
    enabled        = true
    action         = "block"
    filters        = ["http"]
    traffic        = "any(http.request.uri.content_category[*] in {17 85 87 102 157 135 138 180 162 32 169 177 128 15 115 119 124 141 161 2 67 125 133 99})""
    identity       = ""
    device_posture = ""
  }
  ```

## Block unauthorized applications

Note

After seven days, view your [Shadow IT SaaS Analytics](https://developers.cloudflare.com/cloudflare-one/insights/analytics/shadow-it-discovery/) and block additional applications based on what your users are accessing.

To minimize the risk of [shadow IT](https://www.cloudflare.com/learning/access-management/what-is-shadow-it/), some organizations choose to limit their users' access to certain web-based tools and applications. For example, the following policy blocks known AI tools:

* Dashboard

  | Selector | Operator | Value | Action |
  | - | - | - | - |
  | Application | in | *Artificial Intelligence* | Block |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "All-HTTP-Application-Blocklist",
      "description": "Limit access to shadow IT by blocking web-based tools and applications",
      "precedence": 60,
      "enabled": true,
      "action": "block",
      "filters": [
          "http"
      ],
      "traffic": "any(app.type.ids[*] in {25})",
      "identity": "",
      "device_posture": ""
    }'
  ```

* Terraform

  ```tf
  resource "cloudflare_zero_trust_gateway_policy" "all_http_application_blocklist" {
    account_id     = var.cloudflare_account_id
    name           = "All-HTTP-Application-Blocklist"
    description    = "Limit access to shadow IT by blocking web-based tools and applications"
    precedence     = 60
    enabled        = true
    action         = "block"
    filters        = ["http"]
    traffic        = "any(app.type.ids[*] in {25})"
    identity       = ""
    device_posture = ""
  }
  ```

## Check user identity

Configure access on a per user or group basis by adding [identity-based conditions](https://developers.cloudflare.com/cloudflare-one/traffic-policies/identity-selectors/) to your policies.

* Dashboard

  | Selector | Operator | Value | Logic | Action |
  | - | - | - | - | - |
  | Application | in | *Salesforce* | And | Block |
  | User Group Names | in | *Contractors* | | |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Check user identity",
      "description": "Block access to Salesforce by temporary employees and contractors",
      "enabled": true,
      "action": "block",
      "filters": [
          "http"
      ],
      "traffic": "any(app.ids[] in {606})",
      "identity": "any(identity.groups.name[] in {\"Contractors\"})",
      "device_posture": ""
    }'
  ```

## Skip inspection for groups of applications

Certain client applications, such as Zoom or Apple services, rely on certificate pinning. The [TLS decryption](https://developers.cloudflare.com/cloudflare-one/traffic-policies/http-policies/tls-decryption/) performed by Cloudflare Gateway will cause errors when users visit those applications. To avoid this behavior, you must add a Do Not Inspect HTTP policy.

Gateway [evaluates Do Not Inspect policies first](https://developers.cloudflare.com/cloudflare-one/traffic-policies/order-of-enforcement/#http-policies). We recommend moving your Do Not Inspect policies to the top of the list to reduce confusion.

* Dashboard

  | Selector | Operator | Value | Action |
  | - | - | - | - |
  | Application | in | *Do Not Inspect* | Do Not Inspect |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Bypass incompatible applications",
      "description": "Skip TLS decryption for applications that are incompatible with Gateway",
      "enabled": true,
      "action": "off",
      "filters": [
          "http"
      ],
      "traffic": "any(app.type.ids[*] in {16})",
      "identity": "",
      "device_posture": ""
    }'
  ```

Note

You can select either individual applications or the entire Do Not Inspect set, which will update as new applications are added.

## Check device posture

Require devices to have certain software installed or other configuration attributes. For instructions on setting up a device posture check, refer to [Enforce device posture](https://developers.cloudflare.com/cloudflare-one/identity/devices/).

### Enforce a minimum OS version

Perform an [OS version check](https://developers.cloudflare.com/cloudflare-one/identity/devices/warp-client-checks/os-version/) to ensure users are running at least a minimum version.

* Dashboard

  | Selector | Operator | Value | Action |
  | - | - | - | - |
  | Passed Device Posture Checks | in | *Minimum OS version* | Allow |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Require OS version",
      "description": "Perform an OS version check for minimum version",
      "enabled": true,
      "action": "allow",
      "filters": [
          "http"
      ],
      "traffic": "",
      "identity": "",
      "device_posture": "any(device_posture.checks.passed[*] in {\"<POSTURE_CHECK_UUID>\"})"
    }'
  ```

  To get the UUIDs of your device posture checks, use the [List device posture rules](https://developers.cloudflare.com/api/resources/zero_trust/subresources/devices/subresources/posture/methods/list/) endpoint.

### Check for a specific file

Perform a [file check](https://developers.cloudflare.com/cloudflare-one/identity/devices/warp-client-checks/file-check/) to ensure users have a certain file on their device.

Since the file path will be different for each operating system, you can configure a file check for each system and use the **Or** logical operator to only require one of the checks to pass.

* Dashboard

  | Selector | Operator | Value | Logic | Action |
  | - | - | - | - | - |
  | Passed Device Posture Checks | in | *macOS File Check* | Or | Allow |
  | Passed Device Posture Checks | in | *Linux File Check* | | |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Check for specific file",
      "description": "Ensure users have a specific file on their device regardless of operating system",
      "enabled": true,
      "action": "allow",
      "filters": [
          "http"
      ],
      "traffic": "",
      "identity": "",
      "device_posture": "any(device_posture.checks.passed[] in {\"<POSTURE_CHECK_1_UUID>\"}) or any(device_posture.checks.passed[] in {\"<POSTURE_CHECK_2_UUID>\"})"
    }'
  ```

  To get the UUIDs of your device posture checks, use the [List device posture rules](https://developers.cloudflare.com/api/resources/zero_trust/subresources/devices/subresources/posture/methods/list/) endpoint.

## Enforce session duration

[Require users to re-authenticate](https://developers.cloudflare.com/cloudflare-one/team-and-resources/devices/warp/configure-warp/warp-sessions/) after a certain amount of time has elapsed.

## Isolate high risk sites in remote browser

If you are using the [Browser Isolation add-on](https://developers.cloudflare.com/cloudflare-one/remote-browser-isolation/), refer to our list of [common Isolate policies](https://developers.cloudflare.com/cloudflare-one/remote-browser-isolation/isolation-policies/#common-policies).

## Bypass inspection for self-signed certificates

When accessing origin servers with certificates not signed by a public certificate authority, you must bypass TLS decryption.

* Dashboard

  | Selector | Operator | Value | Action |
  | - | - | - | - |
  | Domain | in | `internal.example.com` | Do Not Inspect |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Bypass internal site inspection",
      "description": "Bypass TLS decryption for internal sites with self-signed certificates",
      "enabled": true,
      "action": "off",
      "filters": [
          "http"
      ],
      "traffic": "any(http.conn.domains[*] in {\"internal.example.com\"})",
      "identity": "",
      "device_posture": ""
    }'
  ```

## Block file types

Block the upload or download of files based on their type.

* Dashboard

  | Selector | Operator | Value | Logic | Action |
  | - | - | - | - | - |
  | Upload File Types | in | *Microsoft Office Word Document (docx)* | And | Block |
  | Download File Types | in | *PDF (pdf)* | | |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Block file types",
      "description": "Block the upload or download of files based on their type",
      "enabled": true,
      "action": "block",
      "filters": [
          "http"
      ],
      "traffic": "any(http.upload.file.types[*] in {\"docx\"}) and any(http.download.file.types[*] in {\"pdf\"})",
      "identity": "",
      "device_posture": ""
    }'
  ```

For more information on supported file types, refer to [Download and Upload File Types](https://developers.cloudflare.com/cloudflare-one/traffic-policies/http-policies/#download-and-upload-file-types).

## Isolate or block shadow IT applications

Isolate shadow IT applications discovered by the [Application Library](https://developers.cloudflare.com/cloudflare-one/applications/app-library/) that have not been reviewed yet or are currently under review, and block applications that are not approved by your organization.

For more information on reviewing shadow IT applications, refer to [Review applications](https://developers.cloudflare.com/cloudflare-one/applications/app-library/#review-applications).

### 1. Isolate unreviewed or in review applications

Isolate applications if their approval status is *Unreviewed* or *In review*.

* Dashboard

  | Selector | Operator | Value | Logic | Action |
  | - | - | - | - | - |
  | Application Status | is | *Unreviewed* | Or | Isolate |
  | Application Status | is | *In review* | | |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Isolate unreviewed or in review application status",
      "description": "Isolate Shadow IT applications that have not been reviewed or are in review in the Application Library",
      "enabled": true,
      "action": "isolate",
      "filters": [
          "http"
      ],
      "traffic": "any(app.statuses[*] == \"unreviewed\") or any(app.statuses[*] == \"in review\")",
      "identity": "",
      "device_posture": ""
    }'
  ```

### 2. Block unapproved applications

Block applications if their approval status is *Unapproved*.

* Dashboard

  | Selector | Operator | Value | Action |
  | - | - | - | - |
  | Application Status | is | *Unapproved* | Block |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Block unapproved application status",
      "description": "Block Shadow IT applications that have been marked as unapproved in the Application Library",
      "enabled": true,
      "action": "block",
      "filters": [
          "http"
      ],
      "traffic": "any(app.statuses[*] == \"unapproved\")",
      "identity": "",
      "device_posture": ""
    }'
  ```

## Block Google services

To enable Gateway inspection for Google Drive traffic, you must [add a Cloudflare certificate to Google Drive](https://developers.cloudflare.com/cloudflare-one/team-and-resources/devices/user-side-certificates/manual-deployment/#google-drive).

### Block Google Drive downloads

Block file downloads from Google Drive.

* Dashboard

  | Selector | Operator | Value | Logic | Action |
  | - | - | - | - | - |
  | Application | in | *Google Drive* | And | Block |
  | URL Path & Query | matches regex | `.*(e=download\|export).*` | | |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Block Google Drive downloads",
      "description": "Block file downloads from Google Drive",
      "enabled": true,
      "action": "block",
      "filters": [
          "http"
      ],
      "traffic": "any(app.ids[] in {554}) and http.request.uri.path_and_query matches \".(e=download|export).*\"",
      "identity": "",
      "device_posture": ""
    }'
  ```

### Block Google Drive uploads

Block file uploads from Google Drive.

* Dashboard

  | Selector | Operator | Value | Logic | Action |
  | - | - | - | - | - |
  | Application | in | *Google Drive* | And | Block |
  | Upload Mime Type | matches regex | `.*` | And | |
  | Host | is not | `drivefrontend-pa.clients6.google.com` | | |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Block Google Drive uploads",
      "description": "Block file uploads to Google Drive",
      "enabled": true,
      "action": "block",
      "filters": [
          "http"
      ],
      "traffic": "any(app.ids[] in {554}) and http.upload.mime matches \".\" and not(http.request.host == \"drivefrontend-pa.clients6.google.com\")",
      "identity": "",
      "device_posture": ""
    }'
  ```

### Block Gmail downloads

Block file downloads from Gmail.

* Dashboard

  | Selector | Operator | Value | Logic | Action |
  | - | - | - | - | - |
  | Host | is | `mail-attachment.googleusercontent.com` | And | Block |
  | URL Path & Query | is | `/attachment/u/0` | | |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Block Gmail downloads",
      "description": "Block file downloads from Gmail",
      "enabled": true,
      "action": "block",
      "filters": [
          "http"
      ],
      "traffic": "http.request.host == \"mail-attachment.googleusercontent.com\" and http.request.uri.path_and_query matches \"/attachment/u/0\"",
      "identity": "",
      "device_posture": ""
    }'
  ```

### Block Google Translate proxy

Block use of Google Translate to translate entire webpages.

When translating a website, Google Translate proxies webpages with the `translate.goog` domain. Your users may be able to use this service to bypass other Gateway policies. If you block `translate.goog`, users will still be able to access other Google Translate features.

* Dashboard

  | Selector | Operator | Value | Action |
  | - | - | - | - |
  | Domain | matches regex | `^(.+\.)?translate\.goog$` | Block |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Block Google Translate for websites",
      "description": "Block use of Google Translate to translate entire webpages",
      "enabled": true,
      "action": "block",
      "filters": [
          "http"
      ],
      "traffic": "any(http.request.domains[*] matches \"^(.+\\.)?translate\\.goog$\")",
      "identity": "",
      "device_posture": ""
    }'
  ```

## Filter WebSocket traffic

Gateway does not inspect or log [WebSocket](https://datatracker.ietf.org/doc/html/rfc6455) traffic. Instead, Gateway will only log the HTTP details used to make the WebSocket connection, as well as [network session information](https://developers.cloudflare.com/logs/logpush/logpush-job/datasets/account/zero_trust_network_sessions/). To filter your WebSocket traffic, create a policy with the `101` HTTP response code.

* Dashboard

  | Selector | Operator | Value | Action |
  | - | - | - | - |
  | HTTP Response | is | *101 SWITCHING\_PROTOCOLS* | Allow |

* API

  Required API token permissions

  At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

  * `Zero Trust Write`

  ```bash
  curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/rules" \
    --request POST \
    --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
    --json '{
      "name": "Filter WebSocket",
      "description": "Filter WebSocket traffic with HTTP response code 101",
      "enabled": true,
      "action": "allow",
      "filters": [
          "http"
      ],
      "traffic": "http.response.status_code == 101",
      "identity": "",
      "device_posture": ""
    }'
  ```
